// MyCareerGenie - Job Page Analyzer Content Script
// Detects job listings and provides optimization tools

console.log('MyCareerGenie job analyzer loaded');

let isInitialized = false;
let currentJobData = null;

// Initialize when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initialize);
} else {
  initialize();
}

function initialize() {
  if (isInitialized) return;
  isInitialized = true;
  
  console.log('Initializing job analyzer on:', window.location.href);
  
  // Extract job information
  currentJobData = extractJobData();
  
  if (currentJobData && currentJobData.title) {
    console.log('Job detected:', currentJobData);
    
    // Create floating action panel
    createFloatingPanel();
    
    // Set up observers for dynamic content
    setupObservers();
  }
}

function extractJobData() {
  const hostname = window.location.hostname;
  let jobData = {
    title: '',
    company: '',
    location: '',
    description: '',
    requirements: '',
    benefits: '',
    salary: '',
    jobType: '',
    url: window.location.href,
    source: getJobSource(hostname)
  };

  // LinkedIn job extraction
  if (hostname.includes('linkedin.com')) {
    jobData = extractLinkedInJob();
  }
  // Indeed job extraction
  else if (hostname.includes('indeed.com')) {
    jobData = extractIndeedJob();
  }
  // Glassdoor job extraction
  else if (hostname.includes('glassdoor.com')) {
    jobData = extractGlassdoorJob();
  }
  // Monster job extraction
  else if (hostname.includes('monster.com')) {
    jobData = extractMonsterJob();
  }
  // Walmart Careers job extraction
  else if (hostname.includes('careers.walmart.com')) {
    jobData = extractWalmartJob();
  }
  // Generic job extraction
  else {
    jobData = extractGenericJob();
  }

  // Clean up extracted data
  Object.keys(jobData).forEach(key => {
    if (typeof jobData[key] === 'string') {
      jobData[key] = jobData[key].trim().replace(/\s+/g, ' ');
    }
  });

  return jobData;
}

function extractLinkedInJob() {
  const selectors = {
    title: [
      '.t-24.t-bold',
      '.jobs-unified-top-card__job-title',
      'h1[data-test-id="job-title"]',
      'h1'
    ],
    company: [
      '.jobs-unified-top-card__company-name a',
      '.jobs-unified-top-card__company-name',
      '.job-details-jobs-unified-top-card__company-name a',
      '.job-details-jobs-unified-top-card__company-name'
    ],
    location: [
      '.jobs-unified-top-card__bullet',
      '.job-details-jobs-unified-top-card__bullet'
    ],
    description: [
      '.jobs-box__html-content',
      '.jobs-description-content__text',
      '.jobs-description__container'
    ],
    jobType: [
      '.jobs-unified-top-card__workplace-type',
      '.jobs-unified-top-card__job-insight'
    ]
  };

  return extractUsingSelectors(selectors);
}

function extractIndeedJob() {
  const selectors = {
    title: [
      '[data-testid="jobsearch-JobInfoHeader-title"]',
      '.jobsearch-JobInfoHeader-title',
      'h1'
    ],
    company: [
      '[data-testid="inlineHeader-companyName"]',
      '.icl-u-lg-mr--sm.icl-u-xs-mr--xs',
      'a[data-jk]'
    ],
    location: [
      '[data-testid="job-location"]',
      '.icl-u-colorForeground--secondary.icl-u-xs-mt--xs'
    ],
    description: [
      '[data-testid="jobsearch-jobDescriptionText"]',
      '#jobDescriptionText',
      '.jobsearch-jobDescriptionText'
    ],
    salary: [
      '.icl-u-lg-mr--sm .attribute_snippet',
      '.salary-snippet'
    ]
  };

  return extractUsingSelectors(selectors);
}

function extractGlassdoorJob() {
  const selectors = {
    title: [
      '[data-test="job-title"]',
      '.strong.jobLink',
      'h2'
    ],
    company: [
      '[data-test="employer-name"]',
      '.employerName'
    ],
    location: [
      '[data-test="job-location"]',
      '.location'
    ],
    description: [
      '[data-test="jobDescriptionContainer"]',
      '.jobDescriptionContent',
      '#JobDescriptionContainer'
    ],
    salary: [
      '[data-test="detailSalary"]',
      '.salary'
    ]
  };

  return extractUsingSelectors(selectors);
}

function extractMonsterJob() {
  const selectors = {
    title: [
      '.job-title',
      'h1[data-testid="svx-job-title"]',
      'h1'
    ],
    company: [
      '.company-name',
      '[data-testid="svx-job-company-name"]'
    ],
    location: [
      '.job-location',
      '[data-testid="svx-job-location"]'
    ],
    description: [
      '.job-description',
      '[data-testid="svx-job-description-text"]'
    ]
  };

  return extractUsingSelectors(selectors);
}

function extractWalmartJob() {
  const selectors = {
    title: [
      '[data-automation-id="jobPostingHeader"]',
      '.css-12h2w1r',  // Walmart specific class for job title
      'h1[data-automation-id="jobPostingHeader"]',
      'h1',
      '.job-title',
      '[class*="title"]'
    ],
    company: [
      '[data-automation-id="company"]',
      '.company-name', 
      '.css-1j4221w', // Walmart specific company class
      '[class*="company"]'
    ],
    location: [
      '[data-automation-id="locations"]',
      '[data-automation-id="jobPostingHeaderLocation"]',
      '.css-129m7dg', // Walmart specific location class
      '.location',
      '[class*="location"]'
    ],
    description: [
      '[data-automation-id="jobPostingDescription"]',
      '.css-2tg2r6', // Walmart description container
      '.job-description',
      '.description',
      '.content',
      '[class*="description"]',
      'main'
    ],
    jobType: [
      '[data-automation-id="jobType"]',
      '.job-type',
      '[class*="type"]'
    ],
    requirements: [
      '[data-automation-id="qualifications"]',
      '.qualifications',
      '[class*="requirement"]',
      '[class*="qualification"]'
    ]
  };

  // Try Walmart-specific extraction first
  let jobData = extractUsingSelectors(selectors);
  
  // If company is not found, set to Walmart since we know we're on their site
  if (!jobData.company || jobData.company === 'Unknown Company') {
    jobData.company = 'Walmart';
  }

  // Special handling for Walmart job pages
  if (!jobData.title) {
    // Try alternative title selectors specific to Walmart
    const titleAlternatives = [
      'h1',
      '[role="heading"]',
      '.header h1',
      '.job-header h1',
      'h2'
    ];
    
    for (const selector of titleAlternatives) {
      const element = document.querySelector(selector);
      if (element && element.textContent.trim() && element.textContent.length > 5) {
        jobData.title = element.textContent.trim();
        break;
      }
    }
  }

  // Try to get location from page content if not found
  if (!jobData.location || jobData.location === 'Location not specified') {
    const locationPatterns = [
      /Location[:\s]+([^,\n]+)/i,
      /Based in[:\s]+([^,\n]+)/i,
      /Work[:\s]*Location[:\s]+([^,\n]+)/i
    ];
    
    const pageText = document.body.textContent;
    for (const pattern of locationPatterns) {
      const match = pageText.match(pattern);
      if (match && match[1]) {
        jobData.location = match[1].trim();
        break;
      }
    }
  }

  return jobData;
}

function extractGenericJob() {
  // Enhanced generic selectors for company websites
  const selectors = {
    title: [
      'h1',
      'h2', 
      '.job-title',
      '.position-title',
      '[data-testid*="title"]',
      '[data-automation-id*="title"]', 
      '[class*="title"]',
      '[id*="title"]',
      '[role="heading"]'
    ],
    company: [
      '.company',
      '.employer',
      '.company-name',
      '[data-testid*="company"]',
      '[data-automation-id*="company"]',
      '[class*="company"]',
      '[class*="employer"]'
    ],
    location: [
      '.location',
      '.job-location',
      '[data-testid*="location"]',
      '[data-automation-id*="location"]',
      '[class*="location"]',
      '[class*="address"]'
    ],
    description: [
      '.job-description',
      '.description',
      '.job-details',
      '.job-content',
      '[data-testid*="description"]',
      '[data-automation-id*="description"]',
      '.content',
      'main',
      '[class*="description"]',
      '[class*="content"]'
    ],
    requirements: [
      '.requirements',
      '.qualifications', 
      '[class*="requirement"]',
      '[class*="qualification"]',
      '[data-testid*="requirement"]'
    ]
  };

  let jobData = extractUsingSelectors(selectors);
  
  // Enhanced fallback logic
  if (!jobData.title) {
    // Try to extract from page title
    const pageTitle = document.title;
    if (pageTitle && !pageTitle.toLowerCase().includes('home') && !pageTitle.toLowerCase().includes('careers')) {
      jobData.title = pageTitle.split('|')[0].split('-')[0].trim();
    }
  }
  
  // Try to infer company from hostname if not found
  if (!jobData.company || jobData.company === 'Unknown Company') {
    const hostname = window.location.hostname;
    const domainParts = hostname.replace('www.', '').replace('careers.', '').split('.');
    if (domainParts.length > 1) {
      // Capitalize first letter of domain name
      jobData.company = domainParts[0].charAt(0).toUpperCase() + domainParts[0].slice(1);
    }
  }

  return jobData;
}

function extractUsingSelectors(selectors) {
  const jobData = {
    title: '',
    company: '',
    location: '',
    description: '',
    salary: '',
    jobType: '',
    url: window.location.href,
    source: getJobSource(window.location.hostname)
  };

  Object.keys(selectors).forEach(field => {
    for (const selector of selectors[field]) {
      const element = document.querySelector(selector);
      if (element && element.textContent.trim()) {
        jobData[field] = element.textContent.trim();
        break;
      }
    }
  });

  // Extract additional metadata
  if (!jobData.description) {
    // Try to find the main content area
    const contentSelectors = ['main', '.content', '.job-content', '#content'];
    for (const selector of contentSelectors) {
      const element = document.querySelector(selector);
      if (element && element.textContent.length > 200) {
        jobData.description = element.textContent.trim();
        break;
      }
    }
  }

  return jobData;
}

function getJobSource(hostname) {
  if (hostname.includes('linkedin.com')) return 'LinkedIn';
  if (hostname.includes('indeed.com')) return 'Indeed';
  if (hostname.includes('glassdoor.com')) return 'Glassdoor';
  if (hostname.includes('monster.com')) return 'Monster';
  if (hostname.includes('ziprecruiter.com')) return 'ZipRecruiter';
  if (hostname.includes('careers.walmart.com')) return 'Walmart Careers';
  if (hostname.includes('walmart.com')) return 'Walmart';
  if (hostname.includes('amazon.jobs')) return 'Amazon Jobs';
  if (hostname.includes('google.com/careers')) return 'Google Careers';
  if (hostname.includes('microsoft.com/careers')) return 'Microsoft Careers';
  return 'Company Website';
}

function createFloatingPanel() {
  // Remove existing panel if present
  const existingPanel = document.getElementById('mcg-floating-panel');
  if (existingPanel) {
    existingPanel.remove();
  }

  // Create floating panel
  const panel = document.createElement('div');
  panel.id = 'mcg-floating-panel';
  panel.innerHTML = `
    <div class="mcg-panel-header">
      <div class="mcg-logo">
        <svg width="20" height="20" viewBox="0 0 24 24" fill="none">
          <path d="M12 2L2 7V10C2 16 6 20.5 12 22C18 20.5 22 16 22 10V7L12 2Z" fill="#4F46E5"/>
        </svg>
        <span>MyCareerGenie</span>
      </div>
      <button id="mcg-close-panel" class="mcg-close-btn">&times;</button>
    </div>
    
    <div class="mcg-panel-content">
      <div class="mcg-job-info">
        <h3>${currentJobData.title}</h3>
        <p>${currentJobData.company} • ${currentJobData.location}</p>
      </div>
      
      <div class="mcg-actions">
        <button id="mcg-analyze-btn" class="mcg-btn mcg-btn-primary">
          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <circle cx="11" cy="11" r="8"></circle>
            <path d="m21 21-4.35-4.35"></path>
          </svg>
          Analyze Job
        </button>
        
        <button id="mcg-optimize-btn" class="mcg-btn mcg-btn-secondary">
          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <path d="M14.7 6.3a1 1 0 0 0 0 1.4l1.6 1.6a1 1 0 0 0 1.4 0l3.77-3.77a6 6 0 0 1-7.94 7.94l-6.91 6.91a2.12 2.12 0 0 1-3-3l6.91-6.91a6 6 0 0 1 7.94-7.94l-3.76 3.76z"></path>
          </svg>
          Optimize Resume
        </button>
      </div>
      
      <div id="mcg-status" class="mcg-status hidden">
        <div class="mcg-spinner"></div>
        <span id="mcg-status-text">Processing...</span>
      </div>
      
      <div id="mcg-results" class="mcg-results hidden">
        <div class="mcg-analysis-results"></div>
      </div>
    </div>
  `;

  // Add panel to page
  document.body.appendChild(panel);

  // Setup event listeners
  setupPanelEventListeners();
}

function setupPanelEventListeners() {
  // Close panel
  document.getElementById('mcg-close-panel').addEventListener('click', () => {
    document.getElementById('mcg-floating-panel').remove();
  });

  // Analyze job
  document.getElementById('mcg-analyze-btn').addEventListener('click', analyzeJob);

  // Optimize resume
  document.getElementById('mcg-optimize-btn').addEventListener('click', optimizeResume);
}

async function analyzeJob() {
  showStatus('Analyzing job requirements...');
  
  try {
    // Send message to background script
    const response = await chrome.runtime.sendMessage({
      action: 'analyze_job',
      data: {
        jobDescription: currentJobData.description,
        jobUrl: currentJobData.url,
        jobTitle: currentJobData.title,
        company: currentJobData.company
      }
    });

    if (response && response.success) {
      showResults('Analysis completed successfully!', response.analysis);
    } else {
      showError(response?.error || 'Analysis failed');
    }
  } catch (error) {
    console.error('Job analysis error:', error);
    showError('Analysis failed. Please try again.');
  }
}

async function optimizeResume() {
  showStatus('Optimizing your resume...');
  
  try {
    // First, get user data from background script
    const userDataResponse = await chrome.runtime.sendMessage({
      action: 'get_user_data'
    });

    if (!userDataResponse.success || !userDataResponse.profile) {
      showError('Please sign in to MyCareerGenie dashboard first to optimize your resume.');
      return;
    }

    const userProfile = userDataResponse.profile;
    
    // Check if user is authenticated
    if (!userProfile.id && !userProfile.email) {
      showError('Authentication required. Please visit the MyCareerGenie dashboard and sign in.');
      return;
    }

    // For now, we'll show a helpful message since full CV optimization requires backend setup
    showResults(
      'Resume optimization feature coming soon!', 
      `<div class="mcg-optimization-info">
        <p><strong>What this feature will do:</strong></p>
        <ul>
          <li>• Analyze job requirements vs your resume</li>
          <li>• Suggest specific keywords to add</li>
          <li>• Recommend experience highlights</li>
          <li>• Update your Google Docs resume automatically</li>
        </ul>
        <p><strong>Current status:</strong> Detected user ${userProfile.name || userProfile.email}</p>
        <p><strong>Job analysis:</strong> ${currentJobData.title} at ${currentJobData.company}</p>
        <div style="margin-top: 12px; padding: 8px; background: #f0f9ff; border-radius: 4px;">
          <strong>Tip:</strong> You can manually update your resume by copying key skills and requirements from the job description above.
        </div>
      </div>`
    );

    // Log the attempt for future implementation
    console.log('Resume optimization requested:', {
      user: userProfile.email,
      job: currentJobData.title,
      company: currentJobData.company
    });

  } catch (error) {
    console.error('Resume optimization error:', error);
    showError('Optimization failed. Please make sure you are signed in to MyCareerGenie dashboard.');
  }
}

function showStatus(message) {
  document.getElementById('mcg-status-text').textContent = message;
  document.getElementById('mcg-status').classList.remove('hidden');
  document.getElementById('mcg-results').classList.add('hidden');
}

function showResults(message, data) {
  document.getElementById('mcg-status').classList.add('hidden');
  
  const resultsDiv = document.getElementById('mcg-results');
  resultsDiv.innerHTML = `
    <div class="mcg-success-message">${message}</div>
    ${data && typeof data === 'object' ? formatAnalysisResults(data) : ''}
  `;
  resultsDiv.classList.remove('hidden');
}

function showError(message) {
  document.getElementById('mcg-status').classList.add('hidden');
  
  const resultsDiv = document.getElementById('mcg-results');
  resultsDiv.innerHTML = `<div class="mcg-error-message">${message}</div>`;
  resultsDiv.classList.remove('hidden');
}

function formatAnalysisResults(analysis) {
  if (!analysis || typeof analysis !== 'object') return '';
  
  let html = '<div class="mcg-analysis-details">';
  
  if (analysis.key_skills && analysis.key_skills.length > 0) {
    html += '<div class="mcg-analysis-section">';
    html += '<h4>Key Skills Required:</h4>';
    html += '<ul>';
    analysis.key_skills.forEach(skill => {
      html += `<li>${skill}</li>`;
    });
    html += '</ul>';
    html += '</div>';
  }
  
  if (analysis.optimization_suggestions && analysis.optimization_suggestions.length > 0) {
    html += '<div class="mcg-analysis-section">';
    html += '<h4>Resume Optimization Tips:</h4>';
    html += '<ul>';
    analysis.optimization_suggestions.forEach(tip => {
      html += `<li>${tip}</li>`;
    });
    html += '</ul>';
    html += '</div>';
  }
  
  html += '</div>';
  return html;
}

function setupObservers() {
  // Watch for dynamic content changes
  const observer = new MutationObserver((mutations) => {
    mutations.forEach((mutation) => {
      if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
        // Check if job content has changed
        const newJobData = extractJobData();
        if (newJobData.title && newJobData.title !== currentJobData.title) {
          currentJobData = newJobData;
          
          // Update floating panel if it exists
          const panel = document.getElementById('mcg-floating-panel');
          if (panel) {
            const jobInfo = panel.querySelector('.mcg-job-info');
            if (jobInfo) {
              jobInfo.innerHTML = `
                <h3>${currentJobData.title}</h3>
                <p>${currentJobData.company} • ${currentJobData.location}</p>
              `;
            }
          }
        }
      }
    });
  });

  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
}

// Handle messages from popup or background script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'get_job_data') {
    sendResponse({ success: true, jobData: currentJobData });
  }
  
  if (request.action === 'refresh_analysis') {
    currentJobData = extractJobData();
    sendResponse({ success: true, jobData: currentJobData });
  }
});