// MyCareerGenie - Automatic Form Filler Content Script
// Intelligently fills job application forms with user data

console.log('MyCareerGenie form filler loaded');

let isFormFillerActive = false;
let userProfile = null;
let formFields = [];
let fillButton = null;

// Initialize form filler
initialize();

async function initialize() {
  // Check if this is a job application page
  if (isJobApplicationPage()) {
    console.log('Job application page detected');
    
    // Load user profile
    userProfile = await getUserProfile();
    
    if (userProfile) {
      // Scan for form fields
      scanFormFields();
      
      // Create fill button if forms found
      if (formFields.length > 0) {
        createFillButton();
        
        // Set up observers for dynamic forms
        setupFormObservers();
      }
    }
  }
}

function isJobApplicationPage() {
  const url = window.location.href.toLowerCase();
  const title = document.title.toLowerCase();
  
  // Check URL patterns
  const applicationUrlPatterns = [
    /\/apply/i,
    /\/application/i,
    /\/careers.*apply/i,
    /\/jobs.*apply/i,
    /apply\.html/i,
    /application\.html/i
  ];
  
  // Check for application-related keywords
  const applicationKeywords = [
    'apply', 'application', 'job application', 'career application',
    'submit application', 'apply now', 'employment application'
  ];
  
  const hasApplicationUrl = applicationUrlPatterns.some(pattern => pattern.test(url));
  const hasApplicationTitle = applicationKeywords.some(keyword => 
    title.includes(keyword) || document.body.textContent.toLowerCase().includes(keyword)
  );
  
  // Check for form elements that suggest job applications
  const hasJobForm = document.querySelector([
    'form[action*="apply"]',
    'form[action*="application"]',
    'form[id*="apply"]',
    'form[id*="application"]',
    'form[class*="apply"]',
    'form[class*="application"]',
    '.application-form',
    '.job-application',
    '#application-form',
    '#job-application'
  ].join(','));
  
  return hasApplicationUrl || hasApplicationTitle || hasJobForm;
}

async function getUserProfile() {
  try {
    const response = await chrome.runtime.sendMessage({ action: 'get_user_data' });
    return response.success ? response.profile : null;
  } catch (error) {
    console.error('Failed to load user profile:', error);
    return null;
  }
}

function scanFormFields() {
  formFields = [];
  
  // Find all input fields, textareas, and select elements
  const fields = document.querySelectorAll('input, textarea, select');
  
  fields.forEach(field => {
    const fieldInfo = analyzeField(field);
    if (fieldInfo.type !== 'unknown') {
      formFields.push(fieldInfo);
    }
  });
  
  console.log(`Found ${formFields.length} recognizable form fields`);
}

function analyzeField(element) {
  const field = {
    element: element,
    type: 'unknown',
    label: '',
    required: element.required || element.hasAttribute('required'),
    value: element.value || ''
  };
  
  // Get field label
  field.label = getFieldLabel(element);
  
  // Determine field type based on various attributes
  field.type = determineFieldType(element, field.label);
  
  return field;
}

function getFieldLabel(element) {
  // Try multiple methods to find the field label
  let label = '';
  
  // Method 1: Associated label element
  if (element.id) {
    const labelElement = document.querySelector(`label[for="${element.id}"]`);
    if (labelElement) {
      label = labelElement.textContent.trim();
    }
  }
  
  // Method 2: Parent label
  if (!label) {
    const parentLabel = element.closest('label');
    if (parentLabel) {
      label = parentLabel.textContent.replace(element.value || '', '').trim();
    }
  }
  
  // Method 3: Preceding text or sibling elements
  if (!label) {
    const prevSibling = element.previousElementSibling;
    if (prevSibling && (prevSibling.tagName === 'LABEL' || prevSibling.textContent.trim())) {
      label = prevSibling.textContent.trim();
    }
  }
  
  // Method 4: Placeholder or aria-label
  if (!label) {
    label = element.placeholder || element.getAttribute('aria-label') || '';
  }
  
  // Method 5: Name or id attributes
  if (!label) {
    label = element.name || element.id || '';
  }
  
  return label.replace(/[*:]/g, '').trim();
}

function determineFieldType(element, label) {
  const tagName = element.tagName.toLowerCase();
  const type = element.type ? element.type.toLowerCase() : '';
  const name = element.name ? element.name.toLowerCase() : '';
  const id = element.id ? element.id.toLowerCase() : '';
  const labelLower = label.toLowerCase();
  
  // Combine all text for analysis
  const allText = `${labelLower} ${name} ${id}`.toLowerCase();
  
  // Email field
  if (type === 'email' || /email|e-mail/.test(allText)) {
    return 'email';
  }
  
  // Phone field
  if (type === 'tel' || /phone|tel|mobile|cell/.test(allText)) {
    return 'phone';
  }
  
  // Name fields
  if (/first.*name|fname|given.*name/.test(allText)) {
    return 'firstName';
  }
  if (/last.*name|lname|family.*name|surname/.test(allText)) {
    return 'lastName';
  }
  if (/^name$|full.*name|complete.*name/.test(allText) && !/(company|business|organization)/.test(allText)) {
    return 'fullName';
  }
  
  // Address fields
  if (/address|street/.test(allText)) {
    return 'address';
  }
  if (/city/.test(allText)) {
    return 'city';
  }
  if (/state|province/.test(allText)) {
    return 'state';
  }
  if (/zip|postal|post.*code/.test(allText)) {
    return 'zipCode';
  }
  if (/country/.test(allText)) {
    return 'country';
  }
  
  // Professional fields
  if (/position|title|role|job.*title/.test(allText)) {
    return 'jobTitle';
  }
  if (/company|employer|organization|workplace/.test(allText)) {
    return 'company';
  }
  if (/experience|years.*experience/.test(allText)) {
    return 'experience';
  }
  if (/salary|compensation|pay/.test(allText)) {
    return 'salary';
  }
  
  // Education fields
  if (/education|degree|school|university|college/.test(allText)) {
    return 'education';
  }
  
  // Cover letter / additional info
  if (tagName === 'textarea' && /cover.*letter|additional|why|motivation|summary/.test(allText)) {
    return 'coverLetter';
  }
  
  // LinkedIn profile
  if (/linkedin|profile/.test(allText)) {
    return 'linkedin';
  }
  
  // Website/Portfolio
  if (/website|portfolio|url|link/.test(allText)) {
    return 'website';
  }
  
  return 'unknown';
}

function createFillButton() {
  // Remove existing button
  if (fillButton) {
    fillButton.remove();
  }
  
  fillButton = document.createElement('button');
  fillButton.id = 'mcg-fill-form-btn';
  fillButton.className = 'mcg-fill-form-btn';
  fillButton.innerHTML = `
    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
      <path d="M14.7 6.3a1 1 0 0 0 0 1.4l1.6 1.6a1 1 0 0 0 1.4 0l3.77-3.77a6 6 0 0 1-7.94 7.94l-6.91 6.91a2.12 2.12 0 0 1-3-3l6.91-6.91a6 6 0 0 1 7.94-7.94l-3.76 3.76z"></path>
    </svg>
    Auto-Fill Form
  `;
  
  fillButton.addEventListener('click', fillForm);
  document.body.appendChild(fillButton);
}

async function fillForm() {
  if (!userProfile) {
    showNotification('Please sign in to use auto-fill', 'error');
    return;
  }
  
  fillButton.disabled = true;
  fillButton.textContent = 'Filling...';
  
  let filledCount = 0;
  
  // Fill each field based on its type
  formFields.forEach(field => {
    const value = getValueForFieldType(field.type);
    
    if (value && !field.element.value) {
      fillField(field.element, value);
      filledCount++;
      
      // Add visual indicator
      addFillIndicator(field.element);
    }
  });
  
  // Generate and fill cover letter if found
  const coverLetterFields = formFields.filter(f => f.type === 'coverLetter');
  if (coverLetterFields.length > 0) {
    const coverLetter = await generateCoverLetter();
    if (coverLetter) {
      coverLetterFields.forEach(field => {
        if (!field.element.value) {
          fillField(field.element, coverLetter);
          filledCount++;
          addFillIndicator(field.element);
        }
      });
    }
  }
  
  // Show CV options if there are file upload fields
  await showCVOptions();
  
  // Reset button
  fillButton.disabled = false;
  fillButton.innerHTML = `
    <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
      <path d="M20 6L9 17l-5-5"></path>
    </svg>
    ${filledCount} Fields Filled
  `;
  
  // Log the application attempt
  await logApplicationAttempt(filledCount);
  
  // Show success message
  showNotification(`Successfully filled ${filledCount} fields!`, 'success');
  
  // Reset button text after delay
  setTimeout(() => {
    fillButton.innerHTML = `
      <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
        <path d="M14.7 6.3a1 1 0 0 0 0 1.4l1.6 1.6a1 1 0 0 0 1.4 0l3.77-3.77a6 6 0 0 1-7.94 7.94l-6.91 6.91a2.12 2.12 0 0 1-3-3l6.91-6.91a6 6 0 0 1 7.94-7.94l-3.76 3.76z"></path>
      </svg>
      Auto-Fill Form
    `;
  }, 3000);
}

function getValueForFieldType(type) {
  if (!userProfile) return '';
  
  // Handle both new ResumeData structure and legacy flat structure
  const resumeData = userProfile.resume_data;
  const person = resumeData?.person || {};
  const location = person.location || {};
  const currentExperience = resumeData?.experience?.[0] || {};
  const currentEducation = resumeData?.education?.[0] || {};
  
  const values = {
    // Name fields - try new structure first, fallback to legacy
    firstName: person.name?.split(' ')[0] || userProfile.firstName || userProfile.name?.split(' ')[0] || '',
    lastName: person.name?.split(' ').slice(1).join(' ') || userProfile.lastName || userProfile.name?.split(' ').slice(1).join(' ') || '',
    fullName: person.name || userProfile.name || '',
    
    // Contact info
    email: person.email || userProfile.email || '',
    phone: person.phone || userProfile.phone || '',
    
    // Address fields
    address: location.address || userProfile.address || '',
    city: location.city || userProfile.city || '',
    state: location.state || userProfile.state || '',
    zipCode: location.postal_code || userProfile.zipCode || '',
    country: location.country || userProfile.country || 'United States',
    
    // Professional info
    jobTitle: currentExperience.title || userProfile.currentJobTitle || '',
    company: currentExperience.company || userProfile.currentCompany || '',
    experience: calculateYearsOfExperience(resumeData?.experience) || userProfile.yearsOfExperience || '',
    
    // Education
    education: formatEducation(currentEducation) || userProfile.education || '',
    
    // Links
    linkedin: person.linkedin_url || userProfile.linkedinUrl || '',
    website: person.website_url || userProfile.websiteUrl || userProfile.portfolioUrl || '',
    
    // Salary (legacy field)
    salary: userProfile.desiredSalary || ''
  };
  
  return values[type] || '';
}

// Helper function to calculate years of experience from experience array
function calculateYearsOfExperience(experiences) {
  if (!experiences || experiences.length === 0) return '';
  
  let totalMonths = 0;
  
  experiences.forEach(exp => {
    if (exp.start_date) {
      const startDate = new Date(exp.start_date);
      const endDate = exp.end_date && !exp.is_current ? new Date(exp.end_date) : new Date();
      
      if (!isNaN(startDate.getTime()) && !isNaN(endDate.getTime())) {
        const months = (endDate.getFullYear() - startDate.getFullYear()) * 12 + 
                      (endDate.getMonth() - startDate.getMonth());
        totalMonths += Math.max(0, months);
      }
    }
  });
  
  const years = Math.floor(totalMonths / 12);
  return years > 0 ? years.toString() : '';
}

// Helper function to format education for form fields
function formatEducation(education) {
  if (!education || !education.degree) return '';
  
  let formatted = education.degree;
  if (education.institution) {
    formatted += ` from ${education.institution}`;
  }
  if (education.graduation_date) {
    formatted += ` (${education.graduation_date})`;
  }
  
  return formatted;
}

function fillField(element, value) {
  // Focus the element first
  element.focus();
  
  // Set the value
  element.value = value;
  
  // Trigger change events
  element.dispatchEvent(new Event('input', { bubbles: true }));
  element.dispatchEvent(new Event('change', { bubbles: true }));
  element.dispatchEvent(new Event('blur', { bubbles: true }));
  
  // Add visual feedback
  element.classList.add('mcg-field-filled');
}

function addFillIndicator(element) {
  // Remove existing indicator
  const existingIndicator = element.parentElement.querySelector('.mcg-autofill-indicator');
  if (existingIndicator) {
    existingIndicator.remove();
  }
  
  // Create new indicator
  const indicator = document.createElement('div');
  indicator.className = 'mcg-autofill-indicator';
  indicator.innerHTML = `
    <svg width="12" height="12" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
      <path d="M20 6L9 17l-5-5"></path>
    </svg>
  `;
  
  // Position indicator
  const parent = element.parentElement;
  const originalPosition = getComputedStyle(parent).position;
  if (originalPosition === 'static') {
    parent.style.position = 'relative';
  }
  
  parent.appendChild(indicator);
}

async function generateCoverLetter() {
  try {
    // Get job information from the page
    const jobInfo = extractJobInfoFromPage();
    
    const response = await chrome.runtime.sendMessage({
      action: 'generate_cover_letter',
      data: {
        jobInfo,
        userProfile
      }
    });
    
    return response.success ? response.coverLetter : null;
  } catch (error) {
    console.error('Failed to generate cover letter:', error);
    return null;
  }
}

function extractJobInfoFromPage() {
  // Try to extract job information from the current page
  const jobInfo = {
    title: '',
    company: '',
    description: ''
  };
  
  // Look for job title
  const titleSelectors = ['h1', '.job-title', '.position-title', '[class*="title"]'];
  for (const selector of titleSelectors) {
    const element = document.querySelector(selector);
    if (element && element.textContent.trim()) {
      jobInfo.title = element.textContent.trim();
      break;
    }
  }
  
  // Look for company name
  const companySelectors = ['.company', '.employer', '[class*="company"]'];
  for (const selector of companySelectors) {
    const element = document.querySelector(selector);
    if (element && element.textContent.trim()) {
      jobInfo.company = element.textContent.trim();
      break;
    }
  }
  
  return jobInfo;
}

async function logApplicationAttempt(fieldsFilled = 0) {
  try {
    await chrome.runtime.sendMessage({
      action: 'log_application_attempt',
      data: {
        url: window.location.href,
        timestamp: new Date().toISOString(),
        fieldsFilled: fieldsFilled
      }
    });
  } catch (error) {
    console.error('Failed to log application attempt:', error);
  }
}

function setupFormObservers() {
  // Watch for dynamically added form fields
  const observer = new MutationObserver((mutations) => {
    let needsRescan = false;
    
    mutations.forEach((mutation) => {
      if (mutation.type === 'childList') {
        mutation.addedNodes.forEach((node) => {
          if (node.nodeType === Node.ELEMENT_NODE) {
            const hasFormElements = node.matches && node.matches('input, textarea, select') ||
                                  node.querySelector && node.querySelector('input, textarea, select');
            
            if (hasFormElements) {
              needsRescan = true;
            }
          }
        });
      }
    });
    
    if (needsRescan) {
      setTimeout(() => {
        scanFormFields();
        
        // Update button if needed
        if (formFields.length > 0 && !fillButton) {
          createFillButton();
        } else if (formFields.length === 0 && fillButton) {
          fillButton.remove();
          fillButton = null;
        }
      }, 500);
    }
  });
  
  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
}

function showNotification(message, type = 'info') {
  const notification = document.createElement('div');
  notification.style.cssText = `
    position: fixed;
    top: 20px;
    left: 50%;
    transform: translateX(-50%);
    background: ${type === 'success' ? '#10b981' : '#ef4444'};
    color: white;
    padding: 12px 20px;
    border-radius: 8px;
    font-size: 14px;
    font-weight: 500;
    z-index: 999999;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
  `;
  
  notification.textContent = message;
  document.body.appendChild(notification);
  
  setTimeout(() => {
    notification.remove();
  }, 4000);
}

// Show CV recommendation and generation options
async function showCVOptions() {
  // Check if there are file upload fields that might be for CV/resume
  const fileInputs = document.querySelectorAll('input[type="file"]');
  const resumeInputs = Array.from(fileInputs).filter(input => {
    const label = getFieldLabel(input).toLowerCase();
    const name = (input.name || '').toLowerCase();
    const id = (input.id || '').toLowerCase();
    
    return /resume|cv|curriculum|vitae|document/.test(`${label} ${name} ${id}`);
  });
  
  if (resumeInputs.length === 0) return;
  
  // Create CV options panel
  createCVPanel(resumeInputs[0]);
}

// Create CV recommendation/generation panel
function createCVPanel(targetInput) {
  // Remove existing panel
  const existingPanel = document.querySelector('#mcg-cv-panel');
  if (existingPanel) {
    existingPanel.remove();
  }
  
  const panel = document.createElement('div');
  panel.id = 'mcg-cv-panel';
  panel.className = 'mcg-cv-panel';
  panel.innerHTML = `
    <div class="mcg-cv-header">
      <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
        <polyline points="14,2 14,8 20,8"></polyline>
        <line x1="16" y1="13" x2="8" y2="13"></line>
        <line x1="16" y1="17" x2="8" y2="17"></line>
        <polyline points="10,9 9,9 8,9"></polyline>
      </svg>
      <span>MyCareerGenie CV Assistant</span>
      <button class="mcg-close-btn" onclick="this.closest('#mcg-cv-panel').remove()">×</button>
    </div>
    <div class="mcg-cv-content">
      <p>Get the perfect CV for this job application:</p>
      <div class="mcg-cv-buttons">
        <button id="mcg-recommend-cv" class="mcg-cv-btn mcg-btn-secondary">
          <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <path d="M9 11H5a2 2 0 0 0-2 2v7a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7a2 2 0 0 0-2-2h-4"></path>
            <polyline points="9,11 12,14 15,11"></polyline>
            <line x1="12" y1="2" x2="12" y2="14"></line>
          </svg>
          Recommend Existing CV
        </button>
        <button id="mcg-generate-cv" class="mcg-cv-btn mcg-btn-primary">
          <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <path d="M14.7 6.3a1 1 0 0 0 0 1.4l1.6 1.6a1 1 0 0 0 1.4 0l3.77-3.77a6 6 0 0 1-7.94 7.94l-6.91 6.91a2.12 2.12 0 0 1-3-3l6.91-6.91a6 6 0 0 1 7.94-7.94l-3.76 3.76z"></path>
          </svg>
          Generate New CV
        </button>
      </div>
      <div class="mcg-exaggeration-control" style="display: none;">
        <label for="mcg-exaggeration-slider">Enhancement Level: <span id="mcg-exaggeration-value">0%</span></label>
        <input type="range" id="mcg-exaggeration-slider" min="0" max="100" value="0" step="5">
        <div class="mcg-exaggeration-labels">
          <span>Conservative</span>
          <span>Optimized</span>
        </div>
      </div>
      <div id="mcg-cv-results" style="display: none;">
        <!-- Results will be populated here -->
      </div>
    </div>
  `;
  
  // Position panel near the file input
  const inputRect = targetInput.getBoundingClientRect();
  panel.style.cssText = `
    position: fixed;
    top: ${inputRect.bottom + 10}px;
    left: ${inputRect.left}px;
    background: white;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
    z-index: 999999;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    min-width: 350px;
    max-width: 450px;
  `;
  
  document.body.appendChild(panel);
  
  // Add event listeners
  document.getElementById('mcg-recommend-cv').addEventListener('click', () => recommendCV(targetInput));
  document.getElementById('mcg-generate-cv').addEventListener('click', () => {
    showExaggerationControl();
    generateCV(targetInput);
  });
  
  // Exaggeration slider
  const slider = document.getElementById('mcg-exaggeration-slider');
  const valueDisplay = document.getElementById('mcg-exaggeration-value');
  slider.addEventListener('input', (e) => {
    valueDisplay.textContent = `${e.target.value}%`;
  });
}

// Show exaggeration control
function showExaggerationControl() {
  const control = document.querySelector('.mcg-exaggeration-control');
  if (control) {
    control.style.display = 'block';
  }
}

// Recommend existing CV
async function recommendCV(targetInput) {
  const btn = document.getElementById('mcg-recommend-cv');
  const originalText = btn.innerHTML;
  
  btn.disabled = true;
  btn.innerHTML = 'Finding best CV...';
  
  try {
    const jobInfo = extractJobInfoFromPage();
    
    const response = await chrome.runtime.sendMessage({
      action: 'get_cv_recommendations',
      data: {
        jobDescription: jobInfo.description,
        jobTitle: jobInfo.title,
        company: jobInfo.company,
        jobUrl: window.location.href,
        userId: userProfile?.id
      }
    });
    
    if (response.success && response.recommendations) {
      displayCVRecommendations(response.recommendations, targetInput);
    } else {
      showNotification(response.error || 'No CVs found to recommend', 'error');
    }
  } catch (error) {
    console.error('CV recommendation error:', error);
    showNotification('Failed to get CV recommendations', 'error');
  } finally {
    btn.disabled = false;
    btn.innerHTML = originalText;
  }
}

// Generate new CV
async function generateCV(targetInput) {
  const btn = document.getElementById('mcg-generate-cv');
  const originalText = btn.innerHTML;
  
  btn.disabled = true;
  btn.innerHTML = 'Generating CV...';
  
  try {
    const jobInfo = extractJobInfoFromPage();
    const exaggeration = document.getElementById('mcg-exaggeration-slider')?.value || 0;
    
    const response = await chrome.runtime.sendMessage({
      action: 'generate_cv',
      data: {
        jobDescription: jobInfo.description,
        jobTitle: jobInfo.title,
        company: jobInfo.company,
        jobUrl: window.location.href,
        userId: userProfile?.id,
        exaggeration: parseFloat(exaggeration) / 100 // Convert to 0-1 scale
      }
    });
    
    if (response.success && response.cv) {
      displayGeneratedCV(response, targetInput);
    } else {
      showNotification(response.error || 'Failed to generate CV', 'error');
    }
  } catch (error) {
    console.error('CV generation error:', error);
    showNotification('Failed to generate CV', 'error');
  } finally {
    btn.disabled = false;
    btn.innerHTML = originalText;
  }
}

// Display CV recommendations
function displayCVRecommendations(recommendations, targetInput) {
  const resultsDiv = document.getElementById('mcg-cv-results');
  
  resultsDiv.innerHTML = `
    <div class="mcg-cv-recommendations">
      <h4>Recommended CVs for this job:</h4>
      ${recommendations.map((cv, index) => `
        <div class="mcg-cv-item" data-cv-id="${cv.id}">
          <div class="mcg-cv-info">
            <strong>${cv.name}</strong>
            <span class="mcg-match-score">Match: ${Math.round(cv.match_score * 100)}%</span>
          </div>
          <div class="mcg-cv-meta">
            <span>Updated: ${new Date(cv.updated_at).toLocaleDateString()}</span>
          </div>
          <button class="mcg-download-cv" onclick="downloadCV('${cv.downloadUrl}', '${cv.name}')">
            Download & Use
          </button>
        </div>
      `).join('')}
    </div>
  `;
  
  resultsDiv.style.display = 'block';
}

// Display generated CV
function displayGeneratedCV(cvResponse, targetInput) {
  const resultsDiv = document.getElementById('mcg-cv-results');
  
  resultsDiv.innerHTML = `
    <div class="mcg-cv-generated">
      <h4>✨ Generated CV for this job</h4>
      <div class="mcg-cv-item">
        <div class="mcg-cv-info">
          <strong>Custom CV - ${cvResponse.cv.name}</strong>
          <span class="mcg-generation-badge">✨ AI Generated</span>
        </div>
        <div class="mcg-cv-preview">
          <p>Optimized for: <strong>${extractJobInfoFromPage().title}</strong></p>
          <p>Enhancement level: <strong>${document.getElementById('mcg-exaggeration-value')?.textContent || '0%'}</strong></p>
        </div>
        <div class="mcg-cv-actions">
          <button class="mcg-download-cv" onclick="downloadCV('${cvResponse.downloadUrl}', 'Generated_CV.pdf')">
            Download & Use
          </button>
          ${cvResponse.previewUrl ? `
            <button class="mcg-preview-cv" onclick="window.open('${cvResponse.previewUrl}', '_blank')">
              Preview
            </button>
          ` : ''}
        </div>
      </div>
    </div>
  `;
  
  resultsDiv.style.display = 'block';
}

// Download CV and optionally upload to file input
async function downloadCV(downloadUrl, filename) {
  try {
    // Create download link
    const link = document.createElement('a');
    link.href = downloadUrl;
    link.download = filename;
    link.style.display = 'none';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    
    showNotification(`CV downloaded successfully: ${filename}`, 'success');
    
    // Close the CV panel
    const panel = document.getElementById('mcg-cv-panel');
    if (panel) {
      setTimeout(() => panel.remove(), 2000);
    }
  } catch (error) {
    console.error('Download error:', error);
    showNotification('Failed to download CV', 'error');
  }
}