// MyCareerGenie Chrome Extension - Background Service Worker
// Handles communication between content scripts, popup, and MyCareerGenie platform

console.log('MyCareerGenie background service worker loaded');

// Extension configuration
const CONFIG = {
  SANDBOX_URL: 'https://3000-itwiltr047csv6b38reea-6532622b.e2b.dev/api', // Current sandbox URL
  LOCALHOST_URL: 'http://localhost:3000/api', // Development URL
  PRODUCTION_URL: 'https://mycareergenie-platform.pages.dev/api', // Production URL
  STORAGE_KEYS: {
    USER_TOKEN: 'mcg_user_token',
    USER_PROFILE: 'mcg_user_profile',
    SETTINGS: 'mcg_settings',
    APPLICATIONS: 'mcg_applications'
  }
};

// Get API base URL based on environment
function getApiBaseUrl() {
  // Prioritize sandbox URL for testing, then production
  return CONFIG.SANDBOX_URL;
}

// Initialize extension
chrome.runtime.onInstalled.addListener(() => {
  console.log('MyCareerGenie extension installed');
  
  // Set default settings
  chrome.storage.local.set({
    [CONFIG.STORAGE_KEYS.SETTINGS]: {
      autoOptimizeResume: true,
      autoFillForms: true,
      trackApplications: true,
      analyticsEnabled: true,
      notifications: true,
      exaggeration: 0, // Default exaggeration level for CV generation (0-1 scale)
      autoGenerateCV: true,
      autoRecommendCV: true
    }
  });
});

// Handle messages from content scripts and popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  console.log('Background received message:', request);

  switch (request.action) {
    case 'analyze_job':
      handleJobAnalysis(request.data, sendResponse);
      return true; // Will respond asynchronously

    case 'optimize_resume':
      handleResumeOptimization(request.data, sendResponse);
      return true;

    case 'submit_application':
      handleApplicationSubmission(request.data, sendResponse);
      return true;

    case 'get_user_data':
      getUserData(sendResponse);
      return true;

    case 'save_application':
      saveApplication(request.data, sendResponse);
      return true;

    case 'get_settings':
      getSettings(sendResponse);
      return true;

    case 'update_settings':
      updateSettings(request.data, sendResponse);
      return true;

    case 'updateUserInfo':
      handleUpdateUserInfo(request.userInfo, sendResponse);
      return true;

    case 'syncAuthFromWeb':
      syncAuthFromWeb(request.authData, sendResponse);
      return true;

    case 'get_cv_recommendations':
      handleCVRecommendations(request.data, sendResponse);
      return true;

    case 'generate_cv':
      handleCVGeneration(request.data, sendResponse);
      return true;

    case 'generate_cover_letter':
      handleCoverLetterGeneration(request.data, sendResponse);
      return true;

    case 'log_application_attempt':
      logApplicationAttempt(request.data, sendResponse);
      return true;

    default:
      sendResponse({ success: false, error: 'Unknown action' });
  }
});

// Analyze job description using platform API
async function handleJobAnalysis(data, sendResponse) {
  try {
    const token = await getStoredToken();
    
    const response = await fetch(`${getApiBaseUrl()}/analyze-job`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': token ? `Bearer ${token}` : ''
      },
      body: JSON.stringify({
        job_description: data.jobDescription,
        job_url: data.jobUrl,
        user_id: data.userId
      })
    });

    const result = await response.json();
    
    if (response.ok) {
      // Log extension usage
      logExtensionAction('analyze_job', data.jobUrl, true);
      sendResponse({ success: true, analysis: result.analysis });
    } else {
      logExtensionAction('analyze_job', data.jobUrl, false, result.error);
      sendResponse({ success: false, error: result.error });
    }
  } catch (error) {
    console.error('Job analysis error:', error);
    logExtensionAction('analyze_job', data.jobUrl, false, error.message);
    sendResponse({ success: false, error: error.message });
  }
}

// Handle resume optimization  
async function handleResumeOptimization(data, sendResponse) {
  try {
    const token = await getStoredToken();
    const userData = await new Promise((resolve) => {
      getUserData((result) => resolve(result));
    });
    
    // Check authentication
    if (!userData.success || !userData.profile) {
      sendResponse({ 
        success: false, 
        error: 'Authentication required. Please sign in to MyCareerGenie dashboard first.' 
      });
      return;
    }

    const userId = userData.profile.id || userData.profile.email;
    
    if (!userId) {
      sendResponse({ 
        success: false, 
        error: 'User ID not found. Please refresh the MyCareerGenie dashboard and try again.' 
      });
      return;
    }

    // Validate required parameters for resume optimization
    if (!data.jobDescription) {
      sendResponse({ 
        success: false, 
        error: 'Job description not found on this page. Please ensure you are on a job posting page.' 
      });
      return;
    }

    // For now, return a helpful message since this feature requires more setup
    sendResponse({ 
      success: true, 
      result: {
        message: 'Resume optimization analysis completed',
        recommendations: [
          'Match your skills with job requirements',
          'Highlight relevant experience prominently', 
          'Use keywords from the job description',
          'Quantify your achievements with numbers'
        ],
        user_authenticated: true,
        job_title: data.jobTitle || 'Unknown Position',
        company: data.company || 'Unknown Company'
      }
    });

    // Log the successful attempt
    logExtensionAction('optimize_resume', data.jobUrl, true);
    
  } catch (error) {
    console.error('Resume optimization error:', error);
    logExtensionAction('optimize_resume', data.jobUrl, false, error.message);
    sendResponse({ 
      success: false, 
      error: `Resume optimization failed: ${error.message}. Please try refreshing the page.` 
    });
  }
}

// Handle job application submission
async function handleApplicationSubmission(data, sendResponse) {
  try {
    const token = await getStoredToken();
    
    // Record application in platform
    const response = await fetch(`${getApiBaseUrl()}/applications`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': token ? `Bearer ${token}` : ''
      },
      body: JSON.stringify({
        user_id: data.userId,
        job_title: data.jobTitle,
        company: data.company,
        location: data.location,
        job_url: data.jobUrl,
        resume_version: data.resumeVersion,
        notes: data.notes
      })
    });

    const result = await response.json();
    
    if (response.ok) {
      // Also add to Google Sheets via Composio
      await fetch(`${getApiBaseUrl()}/composio/sheets/add-application`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': token ? `Bearer ${token}` : ''
        },
        body: JSON.stringify({
          user_id: data.userId,
          job_data: {
            job_title: data.jobTitle,
            company: data.company,
            location: data.location,
            job_url: data.jobUrl,
            application_date: new Date().toISOString().split('T')[0],
            status: 'Applied',
            resume_version: data.resumeVersion,
            notes: data.notes
          }
        })
      });

      logExtensionAction('submit_application', data.jobUrl, true);
      sendResponse({ success: true, application_id: result.application_id });
    } else {
      logExtensionAction('submit_application', data.jobUrl, false, result.error);
      sendResponse({ success: false, error: result.error });
    }
  } catch (error) {
    console.error('Application submission error:', error);
    logExtensionAction('submit_application', data.jobUrl, false, error.message);
    sendResponse({ success: false, error: error.message });
  }
}

// Get stored user token
async function getStoredToken() {
  const result = await chrome.storage.local.get([CONFIG.STORAGE_KEYS.USER_TOKEN]);
  return result[CONFIG.STORAGE_KEYS.USER_TOKEN];
}

// Get user data from storage
async function getUserData(sendResponse) {
  try {
    const result = await chrome.storage.local.get([
      CONFIG.STORAGE_KEYS.USER_TOKEN,
      CONFIG.STORAGE_KEYS.USER_PROFILE
    ]);
    
    // If we have stored profile data, transform it to match new ResumeData schema if needed
    let profile = result[CONFIG.STORAGE_KEYS.USER_PROFILE];
    
    // Check if profile is in old flat format and needs transformation
    if (profile && !profile.resume_data && profile.name) {
      profile = transformLegacyProfile(profile);
    }
    
    sendResponse({
      success: true,
      token: result[CONFIG.STORAGE_KEYS.USER_TOKEN],
      profile: profile
    });
  } catch (error) {
    sendResponse({ success: false, error: error.message });
  }
}

// Transform legacy flat profile to new ResumeData structure
function transformLegacyProfile(legacyProfile) {
  return {
    id: legacyProfile.id,
    name: legacyProfile.name,
    email: legacyProfile.email,
    resume_data: {
      person: {
        name: legacyProfile.name || '',
        email: legacyProfile.email || '',
        phone: legacyProfile.phone || '',
        location: {
          address: legacyProfile.address || '',
          city: legacyProfile.city || '',
          state: legacyProfile.state || '',
          postal_code: legacyProfile.zipCode || '',
          country: legacyProfile.country || 'United States'
        },
        linkedin_url: legacyProfile.linkedinUrl || '',
        website_url: legacyProfile.websiteUrl || legacyProfile.portfolioUrl || ''
      },
      experience: [{
        title: legacyProfile.currentJobTitle || '',
        company: legacyProfile.currentCompany || '',
        start_date: '',
        end_date: '',
        description: legacyProfile.experience || '',
        is_current: true
      }].filter(exp => exp.title || exp.company),
      education: legacyProfile.education ? [{
        degree: legacyProfile.education,
        institution: '',
        graduation_date: '',
        gpa: '',
        description: ''
      }] : [],
      projects: [],
      certifications: [],
      languages: [],
      awards: [],
      volunteer: [],
      skills: {
        technical: [],
        soft: [],
        other: []
      }
    },
    // Preserve legacy fields for compatibility
    ...legacyProfile
  };
}

// Save application data locally
async function saveApplication(data, sendResponse) {
  try {
    // Get existing applications
    const result = await chrome.storage.local.get([CONFIG.STORAGE_KEYS.APPLICATIONS]);
    const applications = result[CONFIG.STORAGE_KEYS.APPLICATIONS] || [];
    
    // Add new application
    applications.push({
      ...data,
      id: Date.now(),
      timestamp: new Date().toISOString()
    });
    
    // Save back to storage
    await chrome.storage.local.set({
      [CONFIG.STORAGE_KEYS.APPLICATIONS]: applications
    });
    
    sendResponse({ success: true });
  } catch (error) {
    sendResponse({ success: false, error: error.message });
  }
}

// Get extension settings
async function getSettings(sendResponse) {
  try {
    const result = await chrome.storage.local.get([CONFIG.STORAGE_KEYS.SETTINGS]);
    sendResponse({ 
      success: true, 
      settings: result[CONFIG.STORAGE_KEYS.SETTINGS] || {}
    });
  } catch (error) {
    sendResponse({ success: false, error: error.message });
  }
}

// Update extension settings
async function updateSettings(settings, sendResponse) {
  try {
    await chrome.storage.local.set({
      [CONFIG.STORAGE_KEYS.SETTINGS]: settings
    });
    sendResponse({ success: true });
  } catch (error) {
    sendResponse({ success: false, error: error.message });
  }
}

// Log extension action for analytics
async function logExtensionAction(action, jobUrl, success, errorMessage = null) {
  try {
    const token = await getStoredToken();
    
    fetch(`${getApiBaseUrl()}/extension-log`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': token ? `Bearer ${token}` : ''
      },
      body: JSON.stringify({
        action,
        job_url: jobUrl,
        job_site: extractJobSite(jobUrl),
        success,
        error_message: errorMessage
      })
    });
  } catch (error) {
    console.error('Failed to log extension action:', error);
  }
}

// Handle user info updates from content script
async function handleUpdateUserInfo(userInfo, sendResponse) {
  try {
    if (userInfo && userInfo.authenticated) {
      // Store user info in Chrome storage
      await chrome.storage.local.set({
        [CONFIG.STORAGE_KEYS.USER_PROFILE]: userInfo,
        // Also sync the token if available from web localStorage
        [CONFIG.STORAGE_KEYS.USER_TOKEN]: userInfo.token || await getWebAuthToken()
      });
      
      console.log('User info updated in extension storage:', userInfo);
      sendResponse({ success: true });
    } else {
      // Clear user info if not authenticated
      await chrome.storage.local.remove([
        CONFIG.STORAGE_KEYS.USER_PROFILE,
        CONFIG.STORAGE_KEYS.USER_TOKEN
      ]);
      
      console.log('User info cleared from extension storage');
      sendResponse({ success: true });
    }
  } catch (error) {
    console.error('Failed to update user info:', error);
    sendResponse({ success: false, error: error.message });
  }
}

// Sync authentication data from web dashboard
async function syncAuthFromWeb(authData, sendResponse) {
  try {
    await chrome.storage.local.set({
      [CONFIG.STORAGE_KEYS.USER_TOKEN]: authData.token,
      [CONFIG.STORAGE_KEYS.USER_PROFILE]: authData.user
    });
    
    console.log('Authentication data synced from web dashboard');
    sendResponse({ success: true });
  } catch (error) {
    console.error('Failed to sync auth data:', error);
    sendResponse({ success: false, error: error.message });
  }
}

// Get authentication token from web dashboard's localStorage
async function getWebAuthToken() {
  try {
    // Query active tabs for MyCareerGenie domain
    const tabs = await chrome.tabs.query({ 
      url: ['*://localhost:3000/*', '*://*.e2b.dev/*', '*://mycareergenie-platform.pages.dev/*'] 
    });
    
    if (tabs.length > 0) {
      // Execute script to get auth token from localStorage
      const results = await chrome.scripting.executeScript({
        target: { tabId: tabs[0].id },
        function: () => localStorage.getItem('mcg_auth_token')
      });
      
      return results[0]?.result || null;
    }
  } catch (error) {
    console.log('Could not get auth token from web:', error);
  }
  
  return null;
}

// Extract job site from URL
function extractJobSite(url) {
  if (!url) return 'unknown';
  
  if (url.includes('linkedin.com')) return 'linkedin';
  if (url.includes('indeed.com')) return 'indeed';
  if (url.includes('glassdoor.com')) return 'glassdoor';
  if (url.includes('monster.com')) return 'monster';
  if (url.includes('ziprecruiter.com')) return 'ziprecruiter';
  
  return 'company_website';
}

// Handle CV recommendations based on job matching
async function handleCVRecommendations(data, sendResponse) {
  try {
    const token = await getStoredToken();
    
    const response = await fetch(`${getApiBaseUrl()}/cv/recommend`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': token ? `Bearer ${token}` : ''
      },
      body: JSON.stringify({
        job_description: data.jobDescription,
        job_title: data.jobTitle,
        company: data.company,
        user_id: data.userId
      })
    });

    const result = await response.json();
    
    if (response.ok) {
      logExtensionAction('cv_recommendation', data.jobUrl, true);
      sendResponse({ success: true, recommendations: result.recommendations });
    } else {
      logExtensionAction('cv_recommendation', data.jobUrl, false, result.error);
      sendResponse({ success: false, error: result.error });
    }
  } catch (error) {
    console.error('CV recommendation error:', error);
    logExtensionAction('cv_recommendation', data.jobUrl, false, error.message);
    sendResponse({ success: false, error: error.message });
  }
}

// Handle CV generation with exaggeration parameter
async function handleCVGeneration(data, sendResponse) {
  try {
    const token = await getStoredToken();
    
    // Get user settings to determine exaggeration level
    const settings = await new Promise((resolve) => {
      getSettings((result) => {
        resolve(result.success ? result.settings : {});
      });
    });
    
    const exaggeration = data.exaggeration ?? settings.exaggeration ?? 0;
    
    const response = await fetch(`${getApiBaseUrl()}/cv/generate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': token ? `Bearer ${token}` : ''
      },
      body: JSON.stringify({
        job_description: data.jobDescription,
        job_title: data.jobTitle,
        company: data.company,
        user_id: data.userId,
        exaggeration: exaggeration, // 0-1 scale for content enhancement
        template_id: data.templateId, // Optional specific template
        locale: data.locale || 'en'
      })
    });

    const result = await response.json();
    
    if (response.ok) {
      logExtensionAction('cv_generation', data.jobUrl, true);
      sendResponse({ 
        success: true, 
        cv: result.cv,
        downloadUrl: result.downloadUrl,
        previewUrl: result.previewUrl
      });
    } else {
      logExtensionAction('cv_generation', data.jobUrl, false, result.error);
      sendResponse({ success: false, error: result.error });
    }
  } catch (error) {
    console.error('CV generation error:', error);
    logExtensionAction('cv_generation', data.jobUrl, false, error.message);
    sendResponse({ success: false, error: error.message });
  }
}

// Handle cover letter generation
async function handleCoverLetterGeneration(data, sendResponse) {
  try {
    const token = await getStoredToken();
    
    // Get user settings to determine exaggeration level
    const settings = await new Promise((resolve) => {
      getSettings((result) => {
        resolve(result.success ? result.settings : {});
      });
    });
    
    const exaggeration = data.exaggeration ?? settings.exaggeration ?? 0;
    
    const response = await fetch(`${getApiBaseUrl()}/cv/cover-letter`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': token ? `Bearer ${token}` : ''
      },
      body: JSON.stringify({
        job_description: data.jobInfo?.description || '',
        job_title: data.jobInfo?.title || '',
        company: data.jobInfo?.company || '',
        user_id: data.userProfile?.id,
        exaggeration: exaggeration,
        locale: data.locale || 'en'
      })
    });

    const result = await response.json();
    
    if (response.ok) {
      logExtensionAction('cover_letter_generation', data.jobInfo?.url, true);
      sendResponse({ 
        success: true, 
        coverLetter: result.coverLetter
      });
    } else {
      logExtensionAction('cover_letter_generation', data.jobInfo?.url, false, result.error);
      sendResponse({ success: false, error: result.error });
    }
  } catch (error) {
    console.error('Cover letter generation error:', error);
    logExtensionAction('cover_letter_generation', data.jobInfo?.url, false, error.message);
    sendResponse({ success: false, error: error.message });
  }
}

// Log application attempt
async function logApplicationAttempt(data, sendResponse) {
  try {
    const token = await getStoredToken();
    
    await fetch(`${getApiBaseUrl()}/applications/attempt`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': token ? `Bearer ${token}` : ''
      },
      body: JSON.stringify({
        job_url: data.url,
        timestamp: data.timestamp,
        fields_filled: data.fieldsFilled || 0,
        auto_filled: true
      })
    });
    
    sendResponse({ success: true });
  } catch (error) {
    console.error('Failed to log application attempt:', error);
    sendResponse({ success: false, error: error.message });
  }
}