// MyCareerGenie Chrome Extension - Popup Script

document.addEventListener('DOMContentLoaded', initializePopup);

let currentUser = null;
let currentTab = null;

async function initializePopup() {
  console.log('MyCareerGenie popup initialized');
  
  // Get current tab
  const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
  currentTab = tabs[0];
  
  // Check authentication status
  await checkAuthStatus();
  
  // Check current page for job detection
  await checkCurrentPage();
  
  // Load user stats
  await loadUserStats();
  
  // Load Google integration status
  await loadIntegrationStatus();
  
  // Setup event listeners
  setupEventListeners();
}

async function checkAuthStatus() {
  try {
    // Method 1: Try to get user data from MyCareerGenie website if user is on that domain
    if (currentTab && (currentTab.url.includes('mycareergenie') || 
                      currentTab.url.includes('localhost:3000') || 
                      currentTab.url.includes('e2b.dev'))) {
      try {
        const tabResponse = await chrome.tabs.sendMessage(currentTab.id, { action: 'getUserInfo' });
        if (tabResponse && tabResponse.userInfo && tabResponse.userInfo.authenticated) {
          currentUser = tabResponse.userInfo;
          
          // Sync this auth info to extension storage
          await sendMessageToBackground('updateUserInfo', currentUser);
          
          showLoggedInState();
          return;
        }
      } catch (error) {
        console.log('Could not get user info from current MyCareerGenie tab:', error);
      }
    }
    
    // Method 2: Check all MyCareerGenie tabs
    try {
      const mcgTabs = await chrome.tabs.query({
        url: ['*://localhost:3000/*', '*://*.e2b.dev/*', '*://mycareergenie-platform.pages.dev/*']
      });
      
      for (const tab of mcgTabs) {
        try {
          const tabResponse = await chrome.tabs.sendMessage(tab.id, { action: 'getUserInfo' });
          if (tabResponse && tabResponse.userInfo && tabResponse.userInfo.authenticated) {
            currentUser = tabResponse.userInfo;
            
            // Sync this auth info to extension storage
            await sendMessageToBackground('updateUserInfo', currentUser);
            
            showLoggedInState();
            return;
          }
        } catch (error) {
          console.log(`Could not get user info from tab ${tab.id}:`, error);
        }
      }
    } catch (error) {
      console.log('Could not query MyCareerGenie tabs:', error);
    }
    
    // Method 3: Fallback to extension storage
    const response = await sendMessageToBackground('get_user_data');
    
    if (response.success && response.profile) {
      currentUser = response.profile;
      showLoggedInState();
    } else {
      showLoggedOutState();
    }
  } catch (error) {
    console.error('Auth check failed:', error);
    showLoggedOutState();
  }
}

function showLoggedInState() {
  document.getElementById('logged-out').classList.add('hidden');
  document.getElementById('logged-in').classList.remove('hidden');
  
  if (currentUser) {
    document.getElementById('user-name').textContent = currentUser.name || 'User';
    document.getElementById('user-email').textContent = currentUser.email || '';
  }
}

function showLoggedOutState() {
  document.getElementById('logged-in').classList.add('hidden');
  document.getElementById('logged-out').classList.remove('hidden');
  
  // Update sign-in button to handle new authentication flow
  const signInBtn = document.getElementById('sign-in-btn');
  if (signInBtn) {
    signInBtn.onclick = handleLogin;
  }
}

async function checkCurrentPage() {
  if (!currentTab || !currentTab.url) {
    showNoJobDetected();
    return;
  }
  
  const url = currentTab.url;
  
  // Check if current page is a job listing
  if (isJobPage(url)) {
    updateJobStatus('analyzing', 'Analyzing page...');
    
    try {
      // Inject content script to extract job data
      const results = await chrome.scripting.executeScript({
        target: { tabId: currentTab.id },
        function: extractJobInformation
      });
      
      if (results && results[0] && results[0].result) {
        const jobData = results[0].result;
        showJobDetected(jobData);
        updateJobStatus('success', 'Job detected');
      } else {
        showNoJobDetected();
        updateJobStatus('error', 'No job found');
      }
    } catch (error) {
      console.error('Failed to extract job data:', error);
      showNoJobDetected();
      updateJobStatus('error', 'Analysis failed');
    }
  } else {
    showNoJobDetected();
  }
}

function isJobPage(url) {
  const jobPatterns = [
    /linkedin\.com\/jobs\/view\//i,
    /indeed\.com\/viewjob/i,
    /glassdoor\.com\/job-listing\//i,
    /monster\.com\/job\//i,
    /ziprecruiter\.com\/jobs\//i,
    /\/jobs\//i,
    /\/careers\//i,
    /\/apply/i
  ];
  
  return jobPatterns.some(pattern => pattern.test(url));
}

function extractJobInformation() {
  // This function runs in the context of the job page
  let jobData = {
    title: '',
    company: '',
    location: '',
    description: '',
    url: window.location.href
  };
  
  // LinkedIn selectors
  if (window.location.hostname.includes('linkedin.com')) {
    jobData.title = document.querySelector('.t-24.t-bold')?.textContent?.trim() || 
                   document.querySelector('h1')?.textContent?.trim() || '';
    jobData.company = document.querySelector('.jobs-unified-top-card__company-name')?.textContent?.trim() ||
                     document.querySelector('.job-details-jobs-unified-top-card__company-name')?.textContent?.trim() || '';
    jobData.location = document.querySelector('.jobs-unified-top-card__bullet')?.textContent?.trim() ||
                      document.querySelector('.job-details-jobs-unified-top-card__bullet')?.textContent?.trim() || '';
    jobData.description = document.querySelector('.jobs-box__html-content')?.textContent?.trim() ||
                         document.querySelector('.jobs-description-content__text')?.textContent?.trim() || '';
  }
  
  // Indeed selectors
  else if (window.location.hostname.includes('indeed.com')) {
    jobData.title = document.querySelector('[data-testid="jobsearch-JobInfoHeader-title"]')?.textContent?.trim() ||
                   document.querySelector('h1')?.textContent?.trim() || '';
    jobData.company = document.querySelector('[data-testid="inlineHeader-companyName"]')?.textContent?.trim() ||
                     document.querySelector('.icl-u-lg-mr--sm')?.textContent?.trim() || '';
    jobData.location = document.querySelector('[data-testid="job-location"]')?.textContent?.trim() ||
                      document.querySelector('.icl-u-colorForeground--secondary')?.textContent?.trim() || '';
    jobData.description = document.querySelector('[data-testid="jobsearch-jobDescriptionText"]')?.textContent?.trim() ||
                         document.querySelector('#jobDescriptionText')?.textContent?.trim() || '';
  }
  
  // Glassdoor selectors
  else if (window.location.hostname.includes('glassdoor.com')) {
    jobData.title = document.querySelector('[data-test="job-title"]')?.textContent?.trim() ||
                   document.querySelector('h2')?.textContent?.trim() || '';
    jobData.company = document.querySelector('[data-test="employer-name"]')?.textContent?.trim() ||
                     document.querySelector('.employerName')?.textContent?.trim() || '';
    jobData.location = document.querySelector('[data-test="job-location"]')?.textContent?.trim() ||
                      document.querySelector('.location')?.textContent?.trim() || '';
    jobData.description = document.querySelector('[data-test="jobDescriptionContainer"]')?.textContent?.trim() ||
                         document.querySelector('.jobDescriptionContent')?.textContent?.trim() || '';
  }
  
  // Generic fallbacks
  if (!jobData.title) {
    jobData.title = document.querySelector('h1')?.textContent?.trim() ||
                   document.querySelector('[class*="title"]')?.textContent?.trim() || 
                   document.title || '';
  }
  
  if (!jobData.description) {
    // Look for common description containers
    const descSelectors = [
      '[class*="description"]',
      '[class*="job-desc"]',
      '[id*="description"]',
      '.content',
      'main'
    ];
    
    for (const selector of descSelectors) {
      const element = document.querySelector(selector);
      if (element && element.textContent.length > 100) {
        jobData.description = element.textContent.trim();
        break;
      }
    }
  }
  
  return jobData;
}

function showJobDetected(jobData) {
  document.getElementById('no-job-detected').classList.add('hidden');
  document.getElementById('job-detected').classList.remove('hidden');
  
  document.getElementById('job-title').textContent = jobData.title || 'Unknown Position';
  document.getElementById('job-company').textContent = jobData.company || 'Unknown Company';
  document.getElementById('job-location').textContent = jobData.location || 'Location not specified';
  
  // Store job data for later use
  window.currentJobData = jobData;
}

function showNoJobDetected() {
  document.getElementById('job-detected').classList.add('hidden');
  document.getElementById('no-job-detected').classList.remove('hidden');
  updateJobStatus('error', 'No job detected');
}

function updateJobStatus(type, text) {
  const statusDot = document.querySelector('.status-dot');
  const statusText = document.getElementById('job-status-text');
  
  statusDot.className = `status-dot ${type}`;
  statusText.textContent = text;
}

async function loadUserStats() {
  if (!currentUser) return;
  
  try {
    const response = await fetch(`${getApiUrl()}/analytics/${currentUser.id}`, {
      headers: {
        'Authorization': `Bearer ${await getStoredToken()}`
      }
    });
    
    if (response.ok) {
      const data = await response.json();
      const stats = data.statistics;
      
      document.getElementById('total-apps').textContent = stats.total_applications || 0;
      document.getElementById('interviews').textContent = stats.interviews || 0;
      
      const successRate = stats.total_applications > 0 ? 
        Math.round(((stats.interviews + stats.offers) / stats.total_applications) * 100) : 0;
      document.getElementById('success-rate').textContent = `${successRate}%`;
    }
  } catch (error) {
    console.error('Failed to load user stats:', error);
  }
}

async function loadIntegrationStatus() {
  if (!currentUser) return;
  
  try {
    const response = await fetch(`${getApiUrl()}/composio/status/${currentUser.id}`, {
      headers: {
        'Authorization': `Bearer ${await getStoredToken()}`
      }
    });
    
    if (response.ok) {
      const data = await response.json();
      
      const status = data.connected ? 'connected' : 'disconnected';
      document.getElementById('docs-status').textContent = status;
      document.getElementById('drive-status').textContent = status;
      document.getElementById('sheets-status').textContent = status;
      
      if (data.connected) {
        document.querySelectorAll('.status-badge').forEach(badge => {
          badge.classList.add('connected');
        });
      }
    }
  } catch (error) {
    console.error('Failed to load integration status:', error);
  }
}

function setupEventListeners() {
  // Authentication buttons
  document.getElementById('login-btn').addEventListener('click', handleLogin);
  document.getElementById('register-btn').addEventListener('click', handleRegister);
  
  // Job action buttons
  document.getElementById('analyze-job-btn').addEventListener('click', handleAnalyzeJob);
  document.getElementById('optimize-resume-btn').addEventListener('click', handleOptimizeResume);
  
  // Navigation buttons
  document.getElementById('dashboard-btn').addEventListener('click', handleOpenDashboard);
  document.getElementById('connect-google-btn').addEventListener('click', handleConnectGoogle);
  
  // Settings button
  document.getElementById('settings-btn').addEventListener('click', handleSettings);
  
  // CV Settings
  setupCVSettingsListeners();
}

function setupCVSettingsListeners() {
  // Exaggeration slider
  const exaggerationSlider = document.getElementById('exaggeration-slider');
  const exaggerationDisplay = document.getElementById('exaggeration-display');
  
  if (exaggerationSlider && exaggerationDisplay) {
    exaggerationSlider.addEventListener('input', (e) => {
      exaggerationDisplay.textContent = `${e.target.value}%`;
    });
  }
  
  // Save CV settings button
  const saveSettingsBtn = document.getElementById('save-cv-settings');
  if (saveSettingsBtn) {
    saveSettingsBtn.addEventListener('click', handleSaveCVSettings);
  }
  
  // Load current settings
  loadCVSettings();
}

async function handleLogin() {
  try {
    // Check if user is on MyCareerGenie website already
    const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    const currentTab = tabs[0];
    
    if (currentTab && (currentTab.url.includes('mycareergenie') || 
                      currentTab.url.includes('localhost:3000') || 
                      currentTab.url.includes('e2b.dev'))) {
      // User is already on MyCareerGenie - try to get auth info
      try {
        const response = await chrome.tabs.sendMessage(currentTab.id, { action: 'getUserInfo' });
        if (response && response.userInfo && response.userInfo.authenticated) {
          // User is authenticated on web, sync to extension
          await sendMessageToBackground('updateUserInfo', response.userInfo);
          await checkAuthStatus(); // Refresh auth state
          showNotification('Authentication synced successfully!', 'success');
          return;
        }
      } catch (error) {
        console.log('Could not get auth info from current tab:', error);
      }
      
      // User is on site but not authenticated, show login form
      showNotification('Please sign in to MyCareerGenie first', 'error');
    }
    
    // Open login page in new tab and guide user
    await chrome.tabs.create({
      url: `${getApiUrl().replace('/api', '')}/login`,
      active: true
    });
    
    showNotification('Please sign in on the web page, then click refresh', 'info');
    
    // Add refresh button to popup
    addRefreshButton();
    
  } catch (error) {
    console.error('Login error:', error);
    showNotification('Login failed', 'error');
  }
}

async function handleRegister() {
  try {
    // Check if user is on MyCareerGenie website already
    const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    const currentTab = tabs[0];
    
    if (currentTab && (currentTab.url.includes('mycareergenie') || 
                      currentTab.url.includes('localhost:3000') || 
                      currentTab.url.includes('e2b.dev'))) {
      // User is already on MyCareerGenie - try to get auth info
      try {
        const response = await chrome.tabs.sendMessage(currentTab.id, { action: 'getUserInfo' });
        if (response && response.userInfo && response.userInfo.authenticated) {
          // User is authenticated on web, sync to extension
          await sendMessageToBackground('updateUserInfo', response.userInfo);
          await checkAuthStatus(); // Refresh auth state
          showNotification('Authentication synced successfully!', 'success');
          return;
        }
      } catch (error) {
        console.log('Could not get auth info from current tab:', error);
      }
      
      // User is on site but not authenticated, show signup form
      showNotification('Please create an account on MyCareerGenie first', 'error');
    }
    
    // Open signup page in new tab and guide user
    await chrome.tabs.create({
      url: `${getApiUrl().replace('/api', '')}/signup`,
      active: true
    });
    
    showNotification('Please create account on the web page, then click refresh', 'info');
    
    // Add refresh button to popup
    addRefreshButton();
    
  } catch (error) {
    console.error('Registration error:', error);
    showNotification('Registration failed', 'error');
  }
}

async function handleAnalyzeJob() {
  if (!window.currentJobData) return;
  
  setButtonLoading('analyze-job-btn', true);
  
  try {
    const response = await sendMessageToBackground('analyze_job', {
      jobDescription: window.currentJobData.description,
      jobUrl: window.currentJobData.url,
      userId: currentUser?.id
    });
    
    if (response.success) {
      showNotification('Job analyzed successfully!', 'success');
    } else {
      showNotification(response.error || 'Analysis failed', 'error');
    }
  } catch (error) {
    showNotification('Analysis failed', 'error');
  } finally {
    setButtonLoading('analyze-job-btn', false);
  }
}

async function handleOptimizeResume() {
  if (!window.currentJobData) return;
  
  setButtonLoading('optimize-resume-btn', true);
  
  try {
    const response = await sendMessageToBackground('optimize_resume', {
      jobDescription: window.currentJobData.description,
      jobUrl: window.currentJobData.url,
      jobTitle: window.currentJobData.title,
      userId: currentUser?.id
    });
    
    if (response.success) {
      showNotification('Resume optimization started!', 'success');
    } else {
      showNotification(response.error || 'Optimization failed', 'error');
    }
  } catch (error) {
    showNotification('Optimization failed', 'error');
  } finally {
    setButtonLoading('optimize-resume-btn', false);
  }
}

async function handleOpenDashboard() {
  await chrome.tabs.create({
    url: `${getApiUrl().replace('/api', '')}/dashboard`,
    active: true
  });
}

async function handleConnectGoogle() {
  if (!currentUser) {
    showNotification('Please sign in first', 'error');
    return;
  }
  
  try {
    const response = await fetch(`${getApiUrl()}/composio/connect`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${await getStoredToken()}`
      },
      body: JSON.stringify({
        user_id: currentUser.id,
        services: ['docs', 'drive', 'sheets']
      })
    });
    
    const data = await response.json();
    
    if (data.success && data.auth_url) {
      await chrome.tabs.create({
        url: data.auth_url,
        active: true
      });
    } else {
      showNotification(data.error || 'Connection failed', 'error');
    }
  } catch (error) {
    showNotification('Connection failed', 'error');
  }
}

function handleSettings() {
  // Open settings page or show settings modal
  chrome.tabs.create({
    url: chrome.runtime.getURL('popup/settings.html')
  });
}

// Utility functions
async function sendMessageToBackground(action, data = {}) {
  return new Promise((resolve) => {
    chrome.runtime.sendMessage({ action, data }, resolve);
  });
}

function getApiUrl() {
  // Try sandbox URL first, then fallback to production
  const sandboxUrl = 'https://3000-itwiltr047csv6b38reea-6532622b.e2b.dev/api';
  const productionUrl = 'https://mycareergenie-platform.pages.dev/api';
  const localhostUrl = 'http://localhost:3000/api';
  
  // For now, prioritize sandbox URL for testing
  return sandboxUrl;
}

async function getStoredToken() {
  const result = await chrome.storage.local.get(['mcg_user_token']);
  return result.mcg_user_token;
}

function setButtonLoading(buttonId, loading) {
  const button = document.getElementById(buttonId);
  if (loading) {
    button.disabled = true;
    button.classList.add('loading');
    button.textContent = 'Processing...';
  } else {
    button.disabled = false;
    button.classList.remove('loading');
    // Reset button text - this should be improved to restore original text
  }
}

function addRefreshButton() {
  // Check if refresh button already exists
  if (document.getElementById('refresh-auth-btn')) return;
  
  const loggedOutSection = document.getElementById('logged-out');
  if (!loggedOutSection) return;
  
  const refreshButton = document.createElement('button');
  refreshButton.id = 'refresh-auth-btn';
  refreshButton.innerHTML = '<i class="fas fa-sync-alt"></i> Refresh Authentication';
  refreshButton.className = 'w-full bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-md text-sm font-medium transition-colors mt-2';
  
  refreshButton.onclick = async () => {
    refreshButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Checking...';
    refreshButton.disabled = true;
    
    try {
      await checkAuthStatus();
      if (currentUser) {
        showNotification('Authentication successful!', 'success');
      } else {
        showNotification('Please sign in on MyCareerGenie first', 'error');
      }
    } catch (error) {
      showNotification('Failed to check authentication', 'error');
    }
    
    refreshButton.innerHTML = '<i class="fas fa-sync-alt"></i> Refresh Authentication';
    refreshButton.disabled = false;
  };
  
  loggedOutSection.appendChild(refreshButton);
}

function showNotification(message, type = 'info') {
  // Remove any existing notifications
  const existingNotifications = document.querySelectorAll('.mcg-notification');
  existingNotifications.forEach(n => n.remove());
  
  // Create a simple notification
  const notification = document.createElement('div');
  notification.textContent = message;
  notification.className = 'mcg-notification';
  notification.style.cssText = `
    position: fixed;
    top: 10px;
    right: 10px;
    padding: 12px 16px;
    border-radius: 6px;
    color: white;
    font-size: 13px;
    z-index: 1000;
    animation: slideIn 0.3s ease;
    max-width: 250px;
    word-wrap: break-word;
    ${type === 'success' ? 'background: #10b981;' : 
      type === 'error' ? 'background: #ef4444;' : 
      'background: #3b82f6;'}
  `;
  
  document.body.appendChild(notification);
  
  setTimeout(() => {
    if (notification.parentNode) {
      notification.remove();
    }
  }, 5000);
}

// CV Settings Functions
async function loadCVSettings() {
  try {
    const response = await sendMessageToBackground('get_settings');
    
    if (response.success && response.settings) {
      const settings = response.settings;
      
      // Update UI with current settings
      const exaggerationSlider = document.getElementById('exaggeration-slider');
      const exaggerationDisplay = document.getElementById('exaggeration-display');
      const autoGenerateCV = document.getElementById('auto-generate-cv');
      const autoRecommendCV = document.getElementById('auto-recommend-cv');
      
      if (exaggerationSlider && exaggerationDisplay) {
        const exaggerationValue = (settings.exaggeration || 0) * 100; // Convert from 0-1 to 0-100
        exaggerationSlider.value = exaggerationValue;
        exaggerationDisplay.textContent = `${exaggerationValue}%`;
      }
      
      if (autoGenerateCV) {
        autoGenerateCV.checked = settings.autoGenerateCV !== false; // Default to true
      }
      
      if (autoRecommendCV) {
        autoRecommendCV.checked = settings.autoRecommendCV !== false; // Default to true
      }
    }
  } catch (error) {
    console.error('Failed to load CV settings:', error);
  }
}

async function handleSaveCVSettings() {
  const saveBtn = document.getElementById('save-cv-settings');
  const originalText = saveBtn.innerHTML;
  
  try {
    saveBtn.disabled = true;
    saveBtn.innerHTML = '<div class="spinner"></div> Saving...';
    
    // Get current settings values
    const exaggerationSlider = document.getElementById('exaggeration-slider');
    const autoGenerateCV = document.getElementById('auto-generate-cv');
    const autoRecommendCV = document.getElementById('auto-recommend-cv');
    
    const newSettings = {
      exaggeration: parseFloat(exaggerationSlider.value) / 100, // Convert from 0-100 to 0-1
      autoGenerateCV: autoGenerateCV.checked,
      autoRecommendCV: autoRecommendCV.checked,
      // Preserve other settings
      autoOptimizeResume: true,
      autoFillForms: true,
      trackApplications: true,
      analyticsEnabled: true,
      notifications: true
    };
    
    const response = await sendMessageToBackground('update_settings', newSettings);
    
    if (response.success) {
      showNotification('CV settings saved successfully!', 'success');
    } else {
      showNotification(response.error || 'Failed to save settings', 'error');
    }
  } catch (error) {
    console.error('Failed to save CV settings:', error);
    showNotification('Failed to save settings', 'error');
  } finally {
    saveBtn.disabled = false;
    saveBtn.innerHTML = originalText;
  }
}