// MyCareerGenie - Job Page Analyzer Content Script
// Detects job listings and provides optimization tools

console.log('MyCareerGenie job analyzer loaded');

let isInitialized = false;
let currentJobData = null;

// Initialize when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initialize);
} else {
  initialize();
}

function initialize() {
  if (isInitialized) return;
  isInitialized = true;
  
  console.log('Initializing job analyzer on:', window.location.href);
  
  // Extract job information
  currentJobData = extractJobData();
  
  if (currentJobData && currentJobData.title) {
    console.log('Job detected:', currentJobData);
    
    // Create floating action panel
    createFloatingPanel();
    
    // Set up observers for dynamic content
    setupObservers();
  }
}

function extractJobData() {
  const hostname = window.location.hostname;
  let jobData = {
    title: '',
    company: '',
    location: '',
    description: '',
    requirements: '',
    benefits: '',
    salary: '',
    jobType: '',
    url: window.location.href,
    source: getJobSource(hostname)
  };

  // LinkedIn job extraction
  if (hostname.includes('linkedin.com')) {
    jobData = extractLinkedInJob();
  }
  // Indeed job extraction
  else if (hostname.includes('indeed.com')) {
    jobData = extractIndeedJob();
  }
  // Glassdoor job extraction
  else if (hostname.includes('glassdoor.com')) {
    jobData = extractGlassdoorJob();
  }
  // Monster job extraction
  else if (hostname.includes('monster.com')) {
    jobData = extractMonsterJob();
  }
  // Generic job extraction
  else {
    jobData = extractGenericJob();
  }

  // Clean up extracted data
  Object.keys(jobData).forEach(key => {
    if (typeof jobData[key] === 'string') {
      jobData[key] = jobData[key].trim().replace(/\s+/g, ' ');
    }
  });

  return jobData;
}

function extractLinkedInJob() {
  const selectors = {
    title: [
      '.t-24.t-bold',
      '.jobs-unified-top-card__job-title',
      'h1[data-test-id="job-title"]',
      'h1'
    ],
    company: [
      '.jobs-unified-top-card__company-name a',
      '.jobs-unified-top-card__company-name',
      '.job-details-jobs-unified-top-card__company-name a',
      '.job-details-jobs-unified-top-card__company-name'
    ],
    location: [
      '.jobs-unified-top-card__bullet',
      '.job-details-jobs-unified-top-card__bullet'
    ],
    description: [
      '.jobs-box__html-content',
      '.jobs-description-content__text',
      '.jobs-description__container'
    ],
    jobType: [
      '.jobs-unified-top-card__workplace-type',
      '.jobs-unified-top-card__job-insight'
    ]
  };

  return extractUsingSelectors(selectors);
}

function extractIndeedJob() {
  const selectors = {
    title: [
      '[data-testid="jobsearch-JobInfoHeader-title"]',
      '.jobsearch-JobInfoHeader-title',
      'h1'
    ],
    company: [
      '[data-testid="inlineHeader-companyName"]',
      '.icl-u-lg-mr--sm.icl-u-xs-mr--xs',
      'a[data-jk]'
    ],
    location: [
      '[data-testid="job-location"]',
      '.icl-u-colorForeground--secondary.icl-u-xs-mt--xs'
    ],
    description: [
      '[data-testid="jobsearch-jobDescriptionText"]',
      '#jobDescriptionText',
      '.jobsearch-jobDescriptionText'
    ],
    salary: [
      '.icl-u-lg-mr--sm .attribute_snippet',
      '.salary-snippet'
    ]
  };

  return extractUsingSelectors(selectors);
}

function extractGlassdoorJob() {
  const selectors = {
    title: [
      '[data-test="job-title"]',
      '.strong.jobLink',
      'h2'
    ],
    company: [
      '[data-test="employer-name"]',
      '.employerName'
    ],
    location: [
      '[data-test="job-location"]',
      '.location'
    ],
    description: [
      '[data-test="jobDescriptionContainer"]',
      '.jobDescriptionContent',
      '#JobDescriptionContainer'
    ],
    salary: [
      '[data-test="detailSalary"]',
      '.salary'
    ]
  };

  return extractUsingSelectors(selectors);
}

function extractMonsterJob() {
  const selectors = {
    title: [
      '.job-title',
      'h1[data-testid="svx-job-title"]',
      'h1'
    ],
    company: [
      '.company-name',
      '[data-testid="svx-job-company-name"]'
    ],
    location: [
      '.job-location',
      '[data-testid="svx-job-location"]'
    ],
    description: [
      '.job-description',
      '[data-testid="svx-job-description-text"]'
    ]
  };

  return extractUsingSelectors(selectors);
}

function extractGenericJob() {
  // Generic selectors for company websites
  const selectors = {
    title: [
      'h1',
      '.job-title',
      '.position-title',
      '[class*="title"]',
      '[id*="title"]'
    ],
    company: [
      '.company',
      '.employer',
      '.company-name',
      '[class*="company"]'
    ],
    location: [
      '.location',
      '.job-location',
      '[class*="location"]'
    ],
    description: [
      '.job-description',
      '.description',
      '.job-details',
      '.content',
      'main',
      '[class*="description"]'
    ]
  };

  return extractUsingSelectors(selectors);
}

function extractUsingSelectors(selectors) {
  const jobData = {
    title: '',
    company: '',
    location: '',
    description: '',
    salary: '',
    jobType: '',
    url: window.location.href,
    source: getJobSource(window.location.hostname)
  };

  Object.keys(selectors).forEach(field => {
    for (const selector of selectors[field]) {
      const element = document.querySelector(selector);
      if (element && element.textContent.trim()) {
        jobData[field] = element.textContent.trim();
        break;
      }
    }
  });

  // Extract additional metadata
  if (!jobData.description) {
    // Try to find the main content area
    const contentSelectors = ['main', '.content', '.job-content', '#content'];
    for (const selector of contentSelectors) {
      const element = document.querySelector(selector);
      if (element && element.textContent.length > 200) {
        jobData.description = element.textContent.trim();
        break;
      }
    }
  }

  return jobData;
}

function getJobSource(hostname) {
  if (hostname.includes('linkedin.com')) return 'LinkedIn';
  if (hostname.includes('indeed.com')) return 'Indeed';
  if (hostname.includes('glassdoor.com')) return 'Glassdoor';
  if (hostname.includes('monster.com')) return 'Monster';
  if (hostname.includes('ziprecruiter.com')) return 'ZipRecruiter';
  return 'Company Website';
}

function createFloatingPanel() {
  // Remove existing panel if present
  const existingPanel = document.getElementById('mcg-floating-panel');
  if (existingPanel) {
    existingPanel.remove();
  }

  // Create floating panel
  const panel = document.createElement('div');
  panel.id = 'mcg-floating-panel';
  panel.innerHTML = `
    <div class="mcg-panel-header">
      <div class="mcg-logo">
        <svg width="20" height="20" viewBox="0 0 24 24" fill="none">
          <path d="M12 2L2 7V10C2 16 6 20.5 12 22C18 20.5 22 16 22 10V7L12 2Z" fill="#4F46E5"/>
        </svg>
        <span>MyCareerGenie</span>
      </div>
      <button id="mcg-close-panel" class="mcg-close-btn">&times;</button>
    </div>
    
    <div class="mcg-panel-content">
      <div class="mcg-job-info">
        <h3>${currentJobData.title}</h3>
        <p>${currentJobData.company} • ${currentJobData.location}</p>
      </div>
      
      <div class="mcg-actions">
        <button id="mcg-analyze-btn" class="mcg-btn mcg-btn-primary">
          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <circle cx="11" cy="11" r="8"></circle>
            <path d="m21 21-4.35-4.35"></path>
          </svg>
          Analyze Job
        </button>
        
        <button id="mcg-optimize-btn" class="mcg-btn mcg-btn-secondary">
          <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <path d="M14.7 6.3a1 1 0 0 0 0 1.4l1.6 1.6a1 1 0 0 0 1.4 0l3.77-3.77a6 6 0 0 1-7.94 7.94l-6.91 6.91a2.12 2.12 0 0 1-3-3l6.91-6.91a6 6 0 0 1 7.94-7.94l-3.76 3.76z"></path>
          </svg>
          Optimize Resume
        </button>
      </div>
      
      <div id="mcg-status" class="mcg-status hidden">
        <div class="mcg-spinner"></div>
        <span id="mcg-status-text">Processing...</span>
      </div>
      
      <div id="mcg-results" class="mcg-results hidden">
        <div class="mcg-analysis-results"></div>
      </div>
    </div>
  `;

  // Add panel to page
  document.body.appendChild(panel);

  // Setup event listeners
  setupPanelEventListeners();
}

function setupPanelEventListeners() {
  // Close panel
  document.getElementById('mcg-close-panel').addEventListener('click', () => {
    document.getElementById('mcg-floating-panel').remove();
  });

  // Analyze job
  document.getElementById('mcg-analyze-btn').addEventListener('click', analyzeJob);

  // Optimize resume
  document.getElementById('mcg-optimize-btn').addEventListener('click', optimizeResume);
}

async function analyzeJob() {
  showStatus('Analyzing job requirements...');
  
  try {
    // Send message to background script
    const response = await chrome.runtime.sendMessage({
      action: 'analyze_job',
      data: {
        jobDescription: currentJobData.description,
        jobUrl: currentJobData.url,
        jobTitle: currentJobData.title,
        company: currentJobData.company
      }
    });

    if (response && response.success) {
      showResults('Analysis completed successfully!', response.analysis);
    } else {
      showError(response?.error || 'Analysis failed');
    }
  } catch (error) {
    console.error('Job analysis error:', error);
    showError('Analysis failed. Please try again.');
  }
}

async function optimizeResume() {
  showStatus('Optimizing your resume...');
  
  try {
    // Send message to background script
    const response = await chrome.runtime.sendMessage({
      action: 'optimize_resume',
      data: {
        jobDescription: currentJobData.description,
        jobUrl: currentJobData.url,
        jobTitle: currentJobData.title,
        company: currentJobData.company
      }
    });

    if (response && response.success) {
      showResults('Resume optimized successfully!', 'Your resume has been updated in Google Docs.');
    } else {
      showError(response?.error || 'Optimization failed');
    }
  } catch (error) {
    console.error('Resume optimization error:', error);
    showError('Optimization failed. Please try again.');
  }
}

function showStatus(message) {
  document.getElementById('mcg-status-text').textContent = message;
  document.getElementById('mcg-status').classList.remove('hidden');
  document.getElementById('mcg-results').classList.add('hidden');
}

function showResults(message, data) {
  document.getElementById('mcg-status').classList.add('hidden');
  
  const resultsDiv = document.getElementById('mcg-results');
  resultsDiv.innerHTML = `
    <div class="mcg-success-message">${message}</div>
    ${data && typeof data === 'object' ? formatAnalysisResults(data) : ''}
  `;
  resultsDiv.classList.remove('hidden');
}

function showError(message) {
  document.getElementById('mcg-status').classList.add('hidden');
  
  const resultsDiv = document.getElementById('mcg-results');
  resultsDiv.innerHTML = `<div class="mcg-error-message">${message}</div>`;
  resultsDiv.classList.remove('hidden');
}

function formatAnalysisResults(analysis) {
  if (!analysis || typeof analysis !== 'object') return '';
  
  let html = '<div class="mcg-analysis-details">';
  
  if (analysis.key_skills && analysis.key_skills.length > 0) {
    html += '<div class="mcg-analysis-section">';
    html += '<h4>Key Skills Required:</h4>';
    html += '<ul>';
    analysis.key_skills.forEach(skill => {
      html += `<li>${skill}</li>`;
    });
    html += '</ul>';
    html += '</div>';
  }
  
  if (analysis.optimization_suggestions && analysis.optimization_suggestions.length > 0) {
    html += '<div class="mcg-analysis-section">';
    html += '<h4>Resume Optimization Tips:</h4>';
    html += '<ul>';
    analysis.optimization_suggestions.forEach(tip => {
      html += `<li>${tip}</li>`;
    });
    html += '</ul>';
    html += '</div>';
  }
  
  html += '</div>';
  return html;
}

function setupObservers() {
  // Watch for dynamic content changes
  const observer = new MutationObserver((mutations) => {
    mutations.forEach((mutation) => {
      if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
        // Check if job content has changed
        const newJobData = extractJobData();
        if (newJobData.title && newJobData.title !== currentJobData.title) {
          currentJobData = newJobData;
          
          // Update floating panel if it exists
          const panel = document.getElementById('mcg-floating-panel');
          if (panel) {
            const jobInfo = panel.querySelector('.mcg-job-info');
            if (jobInfo) {
              jobInfo.innerHTML = `
                <h3>${currentJobData.title}</h3>
                <p>${currentJobData.company} • ${currentJobData.location}</p>
              `;
            }
          }
        }
      }
    });
  });

  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
}

// Handle messages from popup or background script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'get_job_data') {
    sendResponse({ success: true, jobData: currentJobData });
  }
  
  if (request.action === 'refresh_analysis') {
    currentJobData = extractJobData();
    sendResponse({ success: true, jobData: currentJobData });
  }
});