// MyCareerGenie Content Script
// Handles communication between web app and extension

console.log('MyCareerGenie Content Script Loaded');

// Listen for messages from extension popup/background
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  console.log('Content script received message:', request);
  
  if (request.action === 'getUserInfo') {
    // Get user info from localStorage
    const userId = localStorage.getItem('mcg_user_id');
    const userEmail = localStorage.getItem('mcg_user_email');
    const userName = localStorage.getItem('mcg_user_name');
    
    const userInfo = userId ? {
      id: userId,
      email: userEmail,
      name: userName,
      authenticated: true,
      created_at: localStorage.getItem('mcg_created_at') || new Date().toISOString()
    } : null;
    
    console.log('Sending user info to extension:', userInfo);
    sendResponse({ userInfo });
    return true; // Indicate async response
  }
  
  if (request.action === 'checkAuth') {
    const isAuthenticated = !!(localStorage.getItem('mcg_user_id') && localStorage.getItem('mcg_user_email'));
    sendResponse({ authenticated: isAuthenticated });
    return true;
  }
});

// Listen for messages from the web page
window.addEventListener('message', (event) => {
  // Only accept messages from same origin
  if (event.origin !== window.location.origin) return;
  
  if (event.data.type === 'MCG_USER_INFO') {
    console.log('Content script received user info from page:', event.data.userInfo);
    
    // Forward user info to extension background/popup
    chrome.runtime.sendMessage({
      action: 'updateUserInfo',
      userInfo: event.data.userInfo
    }).catch(error => {
      console.log('Could not send message to extension (expected if popup is closed):', error);
    });
  }
});

// Inject helper functions into page context
const script = document.createElement('script');
script.textContent = `
  // Helper function for MyCareerGenie web app integration
  window.MyCareerGenie = {
    // Send user info to extension
    sendUserInfoToExtension: function(userInfo) {
      window.postMessage({
        type: 'MCG_USER_INFO',
        userInfo: userInfo
      }, window.location.origin);
    },
    
    // Check if user is authenticated
    isAuthenticated: function() {
      return !!(localStorage.getItem('mcg_user_id') && localStorage.getItem('mcg_user_email'));
    },
    
    // Get user info from localStorage
    getUserInfo: function() {
      const userId = localStorage.getItem('mcg_user_id');
      const userEmail = localStorage.getItem('mcg_user_email');
      const userName = localStorage.getItem('mcg_user_name');
      
      return userId ? {
        id: userId,
        email: userEmail,
        name: userName,
        authenticated: true
      } : null;
    }
  };
  
  console.log('MyCareerGenie helper functions injected');
`;
document.documentElement.appendChild(script);
script.remove();

// Automatically send user info if available
function autoSendUserInfo() {
  const userId = localStorage.getItem('mcg_user_id');
  const userEmail = localStorage.getItem('mcg_user_email');
  const userName = localStorage.getItem('mcg_user_name');
  
  if (userId && userEmail) {
    const userInfo = {
      id: userId,
      email: userEmail,
      name: userName,
      authenticated: true
    };
    
    // Send to extension
    chrome.runtime.sendMessage({
      action: 'updateUserInfo',
      userInfo: userInfo
    }).catch(error => {
      console.log('Could not send user info to extension:', error);
    });
  }
}

// Send user info on page load
document.addEventListener('DOMContentLoaded', autoSendUserInfo);

// Also send immediately if DOM is already ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', autoSendUserInfo);
} else {
  autoSendUserInfo();
}

// Monitor localStorage changes for user authentication
let lastAuthState = localStorage.getItem('mcg_user_id');
let lastTokenState = localStorage.getItem('mcg_auth_token');

setInterval(() => {
  const currentAuthState = localStorage.getItem('mcg_user_id');
  const currentTokenState = localStorage.getItem('mcg_auth_token');
  
  if (currentAuthState !== lastAuthState || currentTokenState !== lastTokenState) {
    lastAuthState = currentAuthState;
    lastTokenState = currentTokenState;
    
    // Send updated user info including token
    autoSendUserInfoWithToken();
  }
}, 1000);

// Enhanced function to send user info with auth token
function autoSendUserInfoWithToken() {
  const userId = localStorage.getItem('mcg_user_id');
  const userEmail = localStorage.getItem('mcg_user_email');
  const userName = localStorage.getItem('mcg_user_name');
  const authToken = localStorage.getItem('mcg_auth_token');
  
  if (userId && userEmail) {
    const userInfo = {
      id: userId,
      email: userEmail,
      name: userName,
      token: authToken,
      authenticated: true,
      synced_at: new Date().toISOString()
    };
    
    // Send to extension with enhanced data
    chrome.runtime.sendMessage({
      action: 'updateUserInfo',
      userInfo: userInfo
    }).catch(error => {
      console.log('Could not send user info to extension:', error);
    });
    
    console.log('Enhanced user info sent to extension:', { userId, userEmail, hasToken: !!authToken });
  } else if (lastAuthState) {
    // User logged out, clear extension storage
    chrome.runtime.sendMessage({
      action: 'updateUserInfo',
      userInfo: null
    }).catch(error => {
      console.log('Could not clear user info from extension:', error);
    });
    
    console.log('User logout detected, clearing extension storage');
  }
}